// Write a value 0x10 to the control register of the RTC
// This will set the clock SQW/OUT to 1Hz and enable the output
// SQWE = 1 and output freq is set as 1Hz.
// OUT  0   0    SQWE   0   0   RS1   RS0
//  0   0   0     1     0   0    0     0 => 0001 0000 (0x10)
//
//	RS1   RS0   SQW/OUT O/P    SQWE   OUT
//   0     0      1Hz           1      X
//   0     1      4.096KHz      1      X
//   1	   0      8.192KHz		1      X
//	 1	   1	 32.768KHz      1      X
//	 X	   X	  0             0      0
//	 X	   X	  1             0      0
	
#include <m8c.h>       																// part specific constants and macros
#include "PSoCAPI.h"    															// PSoC API definitions for all User Modules
#pragma interrupt_handler RTC_Read_ISR												// Declares the interrupt handler for the timer interrupt 

// Declares a type which will be used for reading and writing data from or to the RTC.
// The first byte stores the address of the Seconds Register inside the RTC.  When writing to 
// the RTC, the first byte should be the register address followed by the data.  So having
// the address as a part of the structure allows us to write the address followed by data in
// a single write.
typedef struct RTC_Struct1{
	BYTE Address;
	BYTE Seconds;
	BYTE Minutes;
	BYTE Hours;
	BYTE Day;
	BYTE Date;
	BYTE Month;
	BYTE Year;
}RTC_Struct;

// Prototype declaration
void I2C_Init(void);
void RTC_WriteReg(BYTE RegAddress, BYTE RegData);
void RTC_SetRegAddress(BYTE RegAddress);
void RTC_ReadTime(RTC_Struct* pTime);
void RTC_SetTime(RTC_Struct* pTime);
void Print_Time(RTC_Struct *pTime);

unsigned char bRTC_RegPointer ;														// Contains the address of the register from/to which data has to be read/written
unsigned char fReadRTC;																// Flags used to indicate occurence of various events

RTC_Struct Time;																	// Structure variable to hold the time and date information
RTC_Struct SetTime;																	// Structure variable to hold the time to be set
#define RTC_Addr 0x68																// Constant whose value is RTC's I2C address
#define TIME_LEN sizeof(Time)														// Constant withe value equal to the size of the structure

void main(void)
{
    M8C_EnableGInt;																	// Enables global interrupts
	LCD_Start();																	// Start and Initialize the LCD 
	I2C_Init();																		// Initialize the I2CHW Usermodul
	
	RTC_WriteReg(0x07, 0x10);														// 0x07 is the Adress for the Controlregister and 0x10 the Value for it
	
	SetTime.Address = 0x00;															//Word Address from where data has to be written
	
	// This function is used to set the time and date. 
	SetTime.Seconds = 0x00; 														// Set the seconds as 00 (BCD format)	
	
	SetTime.Minutes = 0x55;															// Set the minute (BCD format) 
	
	SetTime.Hours = 0x64;															// Set the clock in 12 hour mode with PM enabled and time hour value set as 01 PM
																					//  0     12/24     10 Hour/(PM/AM)  10 Hour     Hours
																					//  0     1(12)         1(PM)           0       0 1 0 0 => 0110 0100 (0x64)
	
	SetTime.Day = 0x02;																// Set the day with 1 being sunday and 7 being saturday
	
	SetTime.Date = 0x25;															// Set the Date  (BCD format)
	
	SetTime.Month = 0x07;															// Set the Month (12, BCD format with 1 being January and 12 being December 
	
	SetTime.Year = 0x11;															// Set the year (BCD format)
	
	RTC_SetTime(&SetTime);															// Call the SetTime function to set the time in the RTC chip.
		
	M8C_EnableIntMask(INT_MSK0, INT_MSK0_GPIO);										// Enable GPIO interrupt

	while(1)
	{
		if(fReadRTC)																// If interrupt has occured
		{
			fReadRTC = 0x00;														// Clear the Read Flag
			RTC_ReadTime(&Time);													// Read the time from RTC
		}
	}
}


void I2C_Init(void)																	// Initialize the I2CHW module
{
	I2CHW_Start();																	// Start the I2CHW
	I2CHW_EnableInt();																// Enable the I2C interrupt
	I2CHW_EnableMstr();																// Enable I2CHW module in Master configuration
}

// This function reads the time and date from DS1307 RTC and stores it in the buffer whose 
// address has been passed as an argument
void RTC_ReadTime(RTC_Struct* pTime)
{
	RTC_SetRegAddress(0x00);														// Set the RTC register address to 0x00 which is the seconds register
	I2CHW_fReadBytes(RTC_Addr, (char*)&(pTime->Seconds), 7, I2CHW_CompleteXfer);	// Read 7 bytes of time data from RTC and store at pTime->Seconds
	while(!(I2CHW_bReadI2CStatus() & I2CHW_RD_COMPLETE));							// Wait till Read operation is complete
	I2CHW_ClrRdStatus();															// Clear the Read Complete flag
}

// This function is used to set the date and time.
void RTC_SetTime(RTC_Struct *pTime)
{	
    I2CHW_bWriteBytes(RTC_Addr, (char *)pTime, TIME_LEN, I2CHW_CompleteXfer);		// Write the data into the RTC. The first byte of the pTime structure is the 
																					// RTC register address and is followed by 7 bytes of Time data.
	while(!(I2CHW_bReadI2CStatus() & I2CHW_WR_COMPLETE));							// Wait for the transmission to be completed
	I2CHW_ClrWrStatus();															// Clear the Write complete flag
}

// This function is used to write to a single register in the RTC
void RTC_WriteReg(BYTE RegAddress, BYTE RegData)
{
	BYTE RTCData[2];
	
	RTCData[0] = RegAddress;
	RTCData[1] = RegData;

    I2CHW_bWriteBytes(RTC_Addr, RTCData, 2, I2CHW_CompleteXfer);					//Write the register address and data to the RTC
	while(!(I2CHW_bReadI2CStatus() & I2CHW_WR_COMPLETE));							//Wait for the transmission to be completed
	{
	}
	I2CHW_ClrWrStatus();															// Clear the Write complete flag
}

// This function is used to set the Register Address in the RTC
void RTC_SetRegAddress(BYTE RegAddress)
{
    I2CHW_bWriteBytes(RTC_Addr, &RegAddress, 1, I2CHW_CompleteXfer);				//Write the register address and data to the RTC
	while(!(I2CHW_bReadI2CStatus() & I2CHW_WR_COMPLETE));							//Wait for the transmission to be completed
	{
	}
	I2CHW_ClrWrStatus();															// Clear the Write complete flag
}

// Update the data on the LCD
void Print_Time(RTC_Struct *pTime)
{
	LCD_Position(0,0);																// Position the cursor on the LCD for displaying data
	if(pTime->Hours & 0x40)															// True if 12 hours format is used and false if 24 hours format is used																				
	{																				// Display Hours on the LCD
		LCD_PrHexByte(pTime->Hours & 0x1F);
	}
	else
	{
		LCD_PrHexByte(pTime->Hours & 0x3F);
	}	
	
	LCD_PrCString(":");																// Display the digit separator
	
	LCD_PrHexByte(pTime->Minutes);													// Display Minutes on the LCD
	LCD_PrCString(":");
	
	LCD_PrHexByte(pTime->Seconds);													// Display Seconds on the LCD
		
	if(pTime->Hours & 0x40)															// True if 12 hours format is used and false if 24 hours format is used
	{
			if(pTime->Hours & 0x20)													// Check whether it is PM or AM
			LCD_PrCString(" PM");
		else
			LCD_PrCString(" AM");
	}
	
	// Display the date
	LCD_Position(1,0);
	LCD_PrHexByte(pTime->Date);
	LCD_PrCString("/");
	
	// Display the month
	LCD_PrHexByte(pTime->Month);
	LCD_PrCString("/");
	
	// Display the year
	LCD_PrHexByte(pTime->Year);
	
	// Display Day
	switch(pTime->Day)
	{
		case 1:	LCD_PrCString(" (SUN)");
				break;
				
		case 2:	LCD_PrCString(" (MON)");
				break;
				
		case 3:	LCD_PrCString(" (TUE)");
				break;
				
		case 4:	LCD_PrCString(" (WED)");
				break;
				
		case 5:	LCD_PrCString(" (THU)");
				break;
		
		case 6:	LCD_PrCString(" (FRI)");
				break;
				
		case 7:	LCD_PrCString(" (SAT)");
				break;
				
	}
}

// ISR routine for RTC_Read_Timer
void RTC_Read_ISR(void)
{
	BYTE dummy;
	fReadRTC = 0x01;																	// Event used to read data from RTC 
	dummy = PRT1DR;																		// Read Port1 to enable the ChangeFromRead interrupt to happen
	Print_Time(&Time);																	// Update the LCD
}